/*
 *  LICENSE:
 *  Copyright (c) 2007 Freescale Semiconductor
 *  
 *  Permission is hereby granted, free of charge, to any person 
 *  obtaining a copy of this software and associated documentation 
 *  files (the "Software"), to deal in the Software without 
 *  restriction, including without limitation the rights to use, 
 *  copy, modify, merge, publish, distribute, sublicense, and/or 
 *  sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following 
 *  conditions:
 *  
 *  The above copyright notice and this permission notice 
 *  shall be included in all copies or substantial portions 
 *  of the Software.
 *  
 *  THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
 *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 *  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
 *  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 *  
 *
 *  File: $Id: ppc-axe-driver.h,v 1.4 2007/11/27 20:57:20 r21893 Exp $
 *
 */

/**
 *	\file ppc-axe-driver.h
 *	\brief Main header file for the PowerPC AXE Reference Driver API.
 */
 
/**
 *  \addtogroup PPC_AXE_DRIVER
 *  @{
 *
 */
 
#ifndef __PPC_AXE_DRIVER_H
#define __PPC_AXE_DRIVER_H

/*
 * Implement test mode functionality
 * TODO: Remove this for production release
 */
//#define PPC_AXE_TESTMODE

/* PPC-AXE Error Codes
 */
#define AXE_E_OK			0		/* No error. */
#define AXE_E_PARAM			-1		/* Invalid Parameter */
#define AXE_E_ALLOC			-2		/* Resource allocation failed */
#define AXE_E_TIMEOUT		-3		/* Timeout expired */
#define AXE_E_NOTFOUND		-4		/* Requested item not found / not available */
#define AXE_E_INTERNAL		-100	/* Internal/Unexpected Error */

/*
 * Reserved Process IDs
 */
#define AXE_PID_SCHEDULER		-1
#define AXE_PID_DRIVER			-2
#define AXE_PID_NONE			-3

/**
 *	\brief		Load an AXE task image into memory.
 *				
 *	\param[in]  image - Pointer to the complete executable ELF object file.
 *	\param[in]	priority - Execution priority level of the task.
 *						  
 *	\return		>= 0 - Success.  Return code is the task process id.
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	This function instructs the PPC/AXE server process to load an AXE ELF
 *	image for a Scheduler task into memory.
 */
int AXELoadTask(const void *image, int priority);

/**
 *	\brief		Unload a task image and free all of its resources. 
 *
 *	\param[in]  task_pid - Process identifier of the task to unload.
 *
 *	\return		>= 0 - Success
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	If the task is currently executing, it will be suspended, then any
 *	resources it has dynamically allocated through the PPC driver will be
 *	freed, then the scheduler will be instructed to kill and unload the
 *	process. Finally, the memory allocations for the task code and data spaces
 *	will be released.
 */
int AXEUnloadTask(int task_pid);

/**
 *	\brief		Start/Resume execution of a task.
 *
 *	\param[in]  task_pid - Process identifier of the task to start.
 *
 *	\return		>= 0 - Success
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	Instructs the scheduler to place the specified task into a running state.
 *	The function returns as soon as the state has been updated. The task will
 *	not actually execute code until the scheduler rules allow it time on the
 *	AXE proecessor.
 */
int AXEStartTask(int task_pid);

/**
 *	\brief		Stop execution of a task.
 *
 *	\param[in]  task_pid - Process identifier of the task to suspend.
 *
 *	\return		>= 0 - Success
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	Stops (pauses) execution of a task without unloading or freeing resources.
 */
int AXESuspendTask(int task_pid);

/**
 *	\brief		Send an IPC message to the scheduler or a running task
 *
 *	\param[in]  msg - pointer to a buffer containing the message to be sent.
 *					  The message itself contains addressing parameters.
 *
 *	\return		>= 0 - Success
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	Places a message in the IPC transmit queue for the AXE Scheduler. The
 *	message may be addressed to the scheduler or to a running task. This
 *	function does not block waiting for a reply to the message. Use
 *	AXEReceiveMessageReply to obtain the reply.
 */
int AXESendMessage(void *msg);

/**
 *	\brief		Posts the given message to the host recveive queue
 *
 *	\param[in]  msg - pointer to a buffer containing the message to be sent.
 *					  The message itself contains addressing parameters.
 *
 *	\return		>= 0 - Success
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	Places a message in the PPC/AXE driver's receive queue.  This may be
 *	used to send messages to other AXE client applications or to delay
 *	processing of received messages by returning them to the queue.
 */
int AXESendMessageToHost(void *msg);

/**
 *	\brief		Receive an IPC message from the scheduler or a running task.
 *
 *	\param[in]  timeout - time in milliseconds to wait for a message if no
 *						  message is currently available. A value of 0 here
 *						  indicates an infinite wait time.
 *				pid		- process id of the caller.  Only messages which are
 *						  addressed to this pid or which are replies to 
 *						  messages sent by this pid will be considered.
 *						  A value of 'AXE_PID_NONE' here will cause the 
 *						  function to return the next message in the receive
 *						  queue, regardless of its process id.
 *
 *	\param[out]	msg - pointer to a pointer that is modified by this function
 *					  to point to the received message.
 *
 *	\return		>= 0 - Success
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	Receives a message from the IPC queue. If no messages are present, waits
 *	for one to appear in the queue and then returns that.
 *
 *  This function must be called to allow the AXE driver to process incoming
 *  messages destined for the driver itself or for a PPC application.  It will
 *  process a single message, then return.   If the message is for the driver,
 *	it will take the requested action.  If the message is for an application,
 *  it will be returned in the 'msg' parameter.
 *
 *	Note that message pointers returned by this function will be incremented
 *	by 1 if they are replies to messages sent by the PPC driver or an
 *  application.  The actual message data in that case begins at address
 *	(msg - 1).  The caller may use the least significant bit of the msg
 *	pointer to distinguish message replies from messages originated by the
 *	AXE Scheduler or an AXE task.
 *
 */
int AXEReceiveMessage(void **msg, int pid, unsigned timeout);

/**
 *	\brief		Receive an IPC message reply from the scheduler or a running
 *				task.
 *
 *	\param[in]  timeout - time in milliseconds to wait for a message if no
 *						  message is currently available. A value of 0 here
 *						  indicates an infinite wait time.
 *
 *	\param[out]	msg - pointer to the original sent message.
 *
 *	\return		>= 0 - Success
 *				< 0  - Failed (see PPC-AXE Error Codes)
 *
 *	Waits for a reply to a specific message.  Since the reply uses the same
 *	buffer as the original message, this function does not return that pointer.
 *	It merely returns a success code, after which the caller may examine the
 *	original message buffer for any reply data.
 */
int AXEReceiveMessageReply(void *msg, unsigned timeout);



/** @} PPC_AXE_DRIVER */
#endif /* __PPC_AXE_DRIVER_H */



/*345678901234567890123456789012345678901234567890123456789012345678901234567890
/*       1         2         3         4         5         6         7         8

/**
 *	\brief	
 *	\param[in]  
 *	\param[out]	
 *	\return
 *
 *	
 *
 *  \note
 */
//void func()
//{
//    
//}
